#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "GerransReadJSONInput.h"
#include "SmarTag2_nfctag.h"
#include "stm32l4xx_hal_rtc.h"
#include "SmarTag2.h"
#include "AppSmarTag.h"
#include "SmartNFCType.h"
#include "SmartNFC.h"
#include "main.h"
#include "RTC_M41T80M6F.h"


static const char JSONInputProlog[45] = "\"SPaRASInput.json\"";// name of header to ensure correct profile uploaded

CodeStatus_t findError;//global error flag
const int numberOfSensors = 6;// total number of sensors

typedef enum//Flag for the "find" functions
{
  Search_OK = 0,
  Search_Result_not_found, //Tag Has No Attributes, Attribute Not Found, Attribute Text Missing
  Search_Result_too_long,
  Search_Non_integers, //Unexpected Non-Integers Found
  Search_Tag_not_found,//findAttribute(): Tag Not Found; '>' Not Found; Closing " Not Found; findSection(): startTag Not Found; endTag Not Found; startTag Incomplete; tag syntax incorrect
  Search_Syntax_incorrect
} SearchErrorFlag_t;

//declare Functions
static SearchErrorFlag_t evalDate(const int *dateInts, RTC_DateTypeDef *tempDate);
static SearchErrorFlag_t evalTime(const int *timeInts, RTC_TimeTypeDef *tempTime);
static CodeStatus_t validateThresholds(const char *name, double threshold);

static SearchErrorFlag_t findJSONStartPoint(const char* text, const char* tag, const char startChar, int *dataStartIndex);
static SearchErrorFlag_t findJSONSection(const char* text, const char* tag, char* data, int dataLen);
static SearchErrorFlag_t findJSONStr(const char* text, const char* tag, char* data, int dataLen);
static SearchErrorFlag_t findJSONInt(const char* text, const char* tag, int* data);
static SearchErrorFlag_t findJSONDouble(const char* text, const char* tag, double* data);
static SearchErrorFlag_t findJSONArrayInt(const char* text, const char* tag, int* data, const int dataSize);
//static SearchErrorFlag_t findJSONArrayStr(const char* text, const char* tag, char data[][12], const int dataSize); // not needed function
static void initialiseJSONSensorConfig(struct sensorConfigs *s, char *sensorSettings, WakeTimer_t interruptPeriod);

// General use functions
/**
* @brief  Ensure date array is valid and transfer from int to RTC_DateTypeDef
* @param  const int *dateInts date in array integer form
* @param  RTC_DateTypeDef *tempDate pointer to be updated if date is in valid format
*/
static SearchErrorFlag_t evalDate(const int *dateInts, RTC_DateTypeDef *tempDate){
	SearchErrorFlag_t ret = Search_OK;

	//flag error if date or month value is out of bounds
	if(dateInts[1] < 1 || dateInts[1] > 12) ret = Search_Result_not_found;
	if(dateInts[2]< 1 || dateInts[2] >31) ret = Search_Result_not_found;
	if((dateInts[1] == 4 || dateInts[1] == 6 || dateInts[1] == 9 || dateInts[1] == 11) && (dateInts[2] >30)) ret = Search_Result_not_found;

	// for February flag error in cases of leap year or non leap years
	if(dateInts[1] == 2){
		 if(!(dateInts[0] % 4)){
			 if(dateInts[2] >29) ret = Search_Result_not_found;
		 }else{
			 if(dateInts[2] >28) ret = Search_Result_not_found;
		 }
	}

	// update pointers for date
	if(ret == Search_OK){
		tempDate->Year = (uint8_t) dateInts[0];
		tempDate->Month = (uint8_t) dateInts[1];
		tempDate->Date = (uint8_t) dateInts[2];
		tempDate->WeekDay = RTC_WEEKDAY_MONDAY;
	}else{//flag and print error message in case of error
		if(printText == 1) printf("\tERROR: A date value is out of bounds");
		findError = isError;
	}

	return ret;
}

/**
* @brief  Ensure time array is valid and transfer from int to RTC_TimeTypeDef
* @param  const int timeInts time in array integer form
* @param  RTC_TimeTypeDef *tempTime pointer to be updated if time is in valid format
*/
static SearchErrorFlag_t evalTime(const int *timeInts, RTC_TimeTypeDef *tempTime)
{
	SearchErrorFlag_t ret = Search_OK;

	//flag error if hours, minutes, or seconds are out of bounds
	if(timeInts[0] < 0 || timeInts[0] > 23) ret = Search_Result_not_found;
	if(timeInts[1] < 0 || timeInts[1] > 59) ret = Search_Result_not_found;
	if(timeInts[2] < 0 || timeInts[2] > 59) ret = Search_Result_not_found;

	//update pointers for time
	if(ret == Search_OK){// update object to be returned
		tempTime->Hours = (uint8_t) timeInts[0];
		tempTime->Minutes = (uint8_t) timeInts[1];
		tempTime->Seconds = (uint8_t) timeInts[2];
	}else{//flag and print error message in case of error
		if(printText == 1) printf("\tERROR: A time value is out of bounds");
		findError = isError;
	}
	return ret;
}

/**
* @brief  Ensure thresholds are within valid limits for their assigned sensor
* @param  const char *name name of the sensor to compare threshold to limits
* @param  double threshold the threshold to be validated
* @retval CodeStatus_t ret flag to ensure there have been no errors
*/
static CodeStatus_t validateThresholds(const char *name, double threshold){
	CodeStatus_t ret = isWorking;

	//compares the threshold given to the relevant sensor boundaries
	if(!strcmp(name,"temperature")){
		if(threshold < -10 || threshold > 60) ret = isError;
	} else if (!strcmp(name,"SHT40temperature")){
			if(threshold < -40 || threshold > 125) ret = isError;
	} else if (!strcmp(name,"SHT40humidity")){
			if(threshold < 0 || threshold > 100) ret = isError;
	} else if (!strcmp(name,"pressure")){
		if(threshold < 260 || threshold > 1260) ret = isError;
	} else if (!strcmp(name,"luminosity")){
		if(threshold < 0 || threshold> 30720) ret = isError;
	} else if (!strcmp(name,"imuacc")){
		if(threshold < 0 || threshold > 32000) ret = isError;
	} else if (!strcmp(name,"orientation")){
		if(threshold < 0 || threshold > 6) ret = isError;
	} else if (!strcmp(name,"tiltrec")){
		if(threshold < 0 || threshold > 90) ret = isError;
	}

	return ret;
}

// JSON  Search Functions
/**
* @brief  Find the start of a desired section in JSON file
* @param  const char *text text to be searched through
* @param  const char *tag tag for relevant information to be found
* @param  const char startChar the character that defines the start of the desired section
* @param  const char *dataStartIndex pointer to be updated for start point to be used by parent function
* @retval SearchErrorFlag_t ret flag to ensure there have been no errors within search
*/
static SearchErrorFlag_t findJSONStartPoint(const char* text, const char* tag, const char startChar, int *dataStartIndex){
	SearchErrorFlag_t ret = Search_OK;
	int textLen = strlen(text);
	int buffDataStartIndex = 0;

	char startTag[30] = "\"";// format search into start tag
	strcat(startTag, tag);
	strcat(startTag, "\":");

	char* startTagPoint = strstr(text, startTag);//find beginning  and end of start tag Separately in case of attribute

	if (startTagPoint == NULL) {// return error if startTag or endTag can't be found
		ret = Search_Tag_not_found;
	}else{
		buffDataStartIndex = (startTagPoint-text)+strlen(startTag);//find indexes of the start of the searchString
		while(text[buffDataStartIndex] != startChar){
			if(text[buffDataStartIndex] != ' ' && text[buffDataStartIndex] != '\t' && text[buffDataStartIndex] != '\n' && text[buffDataStartIndex] != '\r'){
				ret = Search_Tag_not_found;
				break;
			}

			if (*dataStartIndex == textLen-1) ret = Search_Tag_not_found;
			buffDataStartIndex +=1;
		}
		buffDataStartIndex +=1;
	}
	*dataStartIndex = buffDataStartIndex;
	 return ret;
}

/**
* @brief  find and return a section of a JSON file
* @param  const char *text text to be searched through
* @param  const char *tag tag denoting relevant section
* @param  char *data pointer of a string to be updated with desired section
* @param  int  datalen maximum length of the section to be returned
* @retval SearchErrorFlag_t ret flag to ensure there have been no errors within search
*/
static SearchErrorFlag_t findJSONSection(const char* text, const char* tag, char* data, int dataLen){

	SearchErrorFlag_t ret = Search_OK;
	int dataStartIndex = 0;
	ret = findJSONStartPoint(text, tag, '{', &dataStartIndex);
	int textLen = strlen(text);

	if(ret == Search_OK){

		int searchCounter = dataStartIndex;
		int inQuotes = 0;
		int bracketLayers = 1;


		while(bracketLayers != 0) {//Copy the data between tags into a the return string

			if(inQuotes == 0){// brackets only count when text is not in quotes
				if (text[searchCounter] == '{'){
					bracketLayers ++;
				}else if (text[searchCounter] == '}'){
					bracketLayers -=1;
				}else if(text[searchCounter] == '"'){// speech marks flip quotes
					inQuotes = 1;
				}
			}else{
				if(text[searchCounter] == '"' && text[searchCounter-1] != '\\'){// speech marks to exit quotes only count without escape character '/'
					inQuotes = 0;
				}
			}

			if(searchCounter == textLen){
				ret = Search_Result_too_long;
				break;
			}

			data[(searchCounter-dataStartIndex)] = text[searchCounter];
			searchCounter++;

			if(searchCounter-dataStartIndex >= dataLen){
				ret = Search_Result_too_long;
				break;
			}

		}
		data[(searchCounter-dataStartIndex)] = '\0';// truncate the string


	}
	if(ret != Search_OK){
		data = NULL;
		findError = isError;
		if(printText == 1) {
			printf("\r\n\t%s:   \tERROR: ", tag);// print the error message

			if (ret == Search_Tag_not_found){
				printf("Tag missing or incorrect syntax");
			}else if(ret == Search_Result_not_found){
				printf("Section text missing");
			}else if(ret == Search_Result_too_long){
				printf("Section text is too long");
			}
		}
	}

	return ret;
}

/**
* @brief  find and return a string within a JSON file
* @param  const char *text text to be searched through
* @param  const char *tag tag denoting relevant string
* @param  char *data pointer of a string to be updated with desired variable
* @param  int  datalen maximum length of the string to be returned
* @retval SearchErrorFlag_t ret flag to ensure there have been no errors within search
*/
static SearchErrorFlag_t findJSONStr(const char* text, const char* tag, char* data, int dataLen){
	SearchErrorFlag_t ret = Search_OK;
	int textLen = strlen(text);
	int dataStartIndex = 0;
	ret = findJSONStartPoint(text, tag, '"', &dataStartIndex);
	int searchCounter = dataStartIndex;
	if(ret == Search_OK){
		int dataNo = 0;
		while(!(text[searchCounter-1] != '\\' && text[searchCounter] == '"')) {//Copy the data between tags into a the return string
			if(searchCounter == textLen){
				ret = Search_Result_too_long;
				break;
			}

			data[dataNo] = text[searchCounter];
			dataNo++;
			searchCounter++;

			if(searchCounter-dataStartIndex >= dataLen){
				ret = Search_Result_too_long;
				break;
			}
		}
		searchCounter++;

		data[dataNo] = '\0';// truncate the string
		if(!strcmp(data, "\0")){// find Error if there is no text between tags
			ret = Search_Result_not_found;
		}

	}

	if(ret == Search_OK && (text[searchCounter] != ',' && text[searchCounter] != '}')){
		while(text[searchCounter] != '}' && text[searchCounter] != ','){
			if(searchCounter == textLen){
				ret = Search_Result_too_long;
				break;
			}
			if(text[searchCounter] != ' ' && text[searchCounter] != '\r' && text[searchCounter] != '\n' && text[searchCounter] != '\t'){
				ret = Search_Syntax_incorrect;
				break;
			}
			searchCounter ++;
		}
	}

	if(ret != Search_OK){
		data = NULL;
		findError = isError;
		if(printText == 1) {
			printf("\r\n\t%s:   \tERROR: ", tag);// print the error message

			if (ret == Search_Tag_not_found){
				printf("Tag missing or incorrect syntax");
			}else if(ret == Search_Result_not_found){
				printf("Section text missing");
			}else if(ret == Search_Result_too_long){
				printf("Section text is too long");
			}else if (ret == Search_Syntax_incorrect){
				printf("Search Syntax Incorrect");
			}
		}
	}

	return ret;
}

/**
* @brief  find and return an integer within a JSON file
* @param  const char *text text to be searched through
* @param  const char *tag tag denoting relevant string
* @param  int *data pointer of an integer to be updated with desired variable
* @retval SearchErrorFlag_t ret flag to ensure there have been no errors within search
*/
static SearchErrorFlag_t findJSONInt(const char* text, const char* tag, int* data){
	SearchErrorFlag_t ret = Search_OK;
	int dataStartIndex;
	char dataBuff[20];
	int gapPresent = 0;
	int dataBuffNo = 0;
	int textLen = strlen(text);

	char startTag[30] = "\"";// format search into start tag
	strcat(startTag, tag);
	strcat(startTag, "\":");
	char* startTagPoint = strstr(text, startTag);//find beginning  and end of start tag Separately in case of attribute
	dataStartIndex = (startTagPoint-text)+strlen(startTag) +1;
	if (startTagPoint == NULL) {// return error if startTag or endTag can't be found
		ret = Search_Tag_not_found;
	}
	if(ret == Search_OK){
		while(text[dataStartIndex] != ',' && text[dataStartIndex] != '}'){
			if(dataStartIndex == textLen){//if end of a number
				ret = Search_Result_too_long;
				break;
			}

			if(gapPresent == 0){
				if((text[dataStartIndex] >= 48 && text[dataStartIndex] <= 57)|| (text[dataStartIndex] == '-')){
					if(dataBuffNo != 0 && (text[dataStartIndex] < 48 || text[dataStartIndex] > 57)){
						ret = Search_Non_integers;
						break;
					}
					dataBuff[dataBuffNo] = text[dataStartIndex];
					dataBuffNo ++;
				}else{
					if(text[dataStartIndex] == ' ' || text[dataStartIndex] == '\r' ||text[dataStartIndex] == '\n' || text[dataStartIndex] == '\t'){
						if(dataBuffNo!= 0){// if a gap is found after data, record gap... if a gap is found before data, dont record gap
							gapPresent = 1;
						}
					}else{//report other non integers
						ret = Search_Non_integers;
						break;
					}
				}
			}

			else{//return error if there are multiple portions of text with a gap between them
				if(text[dataStartIndex] != ' ' && text[dataStartIndex] != '\r' && text[dataStartIndex] != '\n' && text[dataStartIndex] != '\t'){
					ret = Search_Syntax_incorrect;
					break;
				}
			}
			dataStartIndex++;
		}

		dataBuff[dataBuffNo] = '\0';// truncate the string
		if(ret == Search_OK &&!strcmp(dataBuff, "\0")){// find Error if there is no text between tags
			ret = Search_Result_not_found;
		}

	}

	if(ret != Search_OK){
		data = NULL;
		findError = isError;
		if(printText == 1) {
			printf("\r\n\t%s:   \tERROR: ", tag);// print the error message

			if (ret == Search_Tag_not_found){
				printf("Tag missing or incorrect syntax");
			}else if(ret == Search_Result_not_found){
				printf("Section text missing");
			}else if(ret == Search_Result_too_long){
				printf("Section text is too long");
			}else if (ret == Search_Non_integers){
				printf("Unexpected Non-Integers Found");
			}else if (ret == Search_Syntax_incorrect){
				printf("Search Syntax Incorrect");
			}
		}
	}else{
		*data = atoi(dataBuff);
	}

	return ret;
}

/**
* @brief  find and return a double within a JSON file
* @param  const char *text text to be searched through
* @param  const char *tag tag denoting relevant string
* @param  double *data pointer of a double to be updated with desired variable
* @retval SearchErrorFlag_t ret flag to ensure there have been no errors within search
*/
static SearchErrorFlag_t findJSONDouble(const char* text, const char* tag, double* data){
	SearchErrorFlag_t ret = Search_OK;
	int dataStartIndex;
	char dataBuff[20];
	int gapPresent = 0;
	int decimalPresent = 0;
	int dataBuffNo = 0;
	int textLen = strlen(text);

	char startTag[30] = "\"";// format search into start tag
	strcat(startTag, tag);
	strcat(startTag, "\":");
	char* startTagPoint = strstr(text, startTag);//find beginning  and end of start tag Separately in case of attribute
	dataStartIndex = (startTagPoint-text)+strlen(startTag) +1;
	if (startTagPoint == NULL) {// return error if startTag or endTag can't be found
		ret = Search_Tag_not_found;
	}
	if(ret == Search_OK){
		while(text[dataStartIndex] != ',' && text[dataStartIndex] != '}'){
			if(dataStartIndex == textLen){//if end of a number
				ret = Search_Result_too_long;
				break;
			}

			if(gapPresent == 0){
				if((text[dataStartIndex] >= 48 && text[dataStartIndex] <= 57)||(text[dataStartIndex] == '-')||(text[dataStartIndex] == '.')){

					if(text[dataStartIndex] == '-' && dataBuffNo != 0){
						ret = Search_Non_integers;
						break;

					}
					if(text[dataStartIndex] == '.' && decimalPresent == 1){
							ret = Search_Non_integers;
							break;
					}
					dataBuff[dataBuffNo] = text[dataStartIndex];
					dataBuffNo ++;
				}else if(text[dataStartIndex] == ' ' || text[dataStartIndex] == '\r' ||text[dataStartIndex] == '\n' || text[dataStartIndex] == '\t'){
						if(dataBuffNo!= 0){// if a gap is found after data, record gap... if a gap is found before data, dont record gap
							gapPresent = 1;
						}
				}else{//report other non integers
					ret = Search_Non_integers;
					break;
				}

			}else{//return error if there are multiple portions of text with a gap between them
				if(text[dataStartIndex] != ' ' && text[dataStartIndex] != '\r' && text[dataStartIndex] != '\n' && text[dataStartIndex] != '\t'){
					ret = Search_Syntax_incorrect;
					break;
				}
			}
			dataStartIndex++;
		}

		dataBuff[dataBuffNo] = '\0';// truncate the string
		if(ret == Search_OK && !strcmp(dataBuff, "\0")){// find Error if there is no text between tags
			ret = Search_Result_not_found;
		}

	}

	if(ret != Search_OK){
		data = NULL;
		findError = isError;
		if(printText == 1) {
			printf("\r\n\t%s:   \tERROR: ", tag);// print the error message

			if (ret == Search_Tag_not_found){
				printf("Tag missing or incorrect syntax");
			}else if(ret == Search_Result_not_found){
				printf("Section text missing");
			}else if(ret == Search_Result_too_long){
				printf("Section text is too long");
			}else if (ret == Search_Non_integers){
				printf("Unexpected Non-Integers Found");
			}else if (ret == Search_Syntax_incorrect){
				printf("Search Syntax Incorrect");
			}
		}
	}else{
		*data = atof(dataBuff);
	}

	return ret;
}

/**
* @brief  find and return an array of integers within a JSON file
* @param  const char *text text to be searched through
* @param  const char *tag tag denoting relevant string
* @param  int *data pointer of an array of integers to be updated with desired variables
* @retval SearchErrorFlag_t ret flag to ensure there have been no errors within search
*/
static SearchErrorFlag_t findJSONArrayInt(const char* text, const char* tag, int* data, const int dataSize){
	SearchErrorFlag_t ret = Search_OK;
	int searchCounter = 0;
	char dataBuff[10];
	int textLen = strlen(text);

	ret = findJSONStartPoint(text, tag, '[', &searchCounter);

	if(ret >= Search_OK){
		int arrayNo =0;

		while(text[searchCounter] !=  ']' && ret == Search_OK) {//strip data into a new array
			strcpy(dataBuff,NULL);// reset variables used in each loop
			int dataBuffNo = 0;
			int gapPresent = 0;

			if(arrayNo == dataSize){// end search is there is more than the expected number of elements
				ret = Search_Result_too_long;
				break;
			}

			while(text[searchCounter] != ',' && text[searchCounter] != ']'){
				if(searchCounter == textLen){//if end of a number
					ret = Search_Result_too_long;
					break;
				}

				if(gapPresent == 0){
					if((text[searchCounter] <48 || text[searchCounter] > 57)){
						if(text[searchCounter] == ' ' || text[searchCounter] == '\r' ||text[searchCounter] == '\n' || text[searchCounter] == '\t'){
							if(dataBuffNo!= 0){// if a gap is found after data, record gap... if a gap is found before data, dont record gap
								gapPresent = 1;
							}
						}else{//report other non integers
							if(!(dataBuffNo == 0 && text[searchCounter] == 45)){// if the first character is a - allow it to enable negative numbers
								ret = Search_Non_integers;
								break;
							}
						}
					}else{
						dataBuff[dataBuffNo] = text[searchCounter];
						dataBuffNo ++;
					}

				}else{//return error if there are multiple portions of text with a gap between them
					if(text[searchCounter] != ' ' && text[searchCounter] != '\r' && text[searchCounter] != '\n' && text[searchCounter] != '\t'){
						ret = Search_Syntax_incorrect;
						break;
					}
				}
				searchCounter++;
			}

			if(ret != Search_OK){
				break;
			}
			if(!strcmp(dataBuff, "\0")){// if number is empty
				ret = Search_Result_not_found;// result not found
				break;
			}
			dataBuff[dataBuffNo] = '\0';
			data[arrayNo] = atoi(dataBuff);//if number is not empty add number to array
			arrayNo ++;// increase non temporary variables
			if(text[searchCounter] != ']'){
				searchCounter ++;
			}
		}
		searchCounter ++;

		if(ret == Search_OK && (text[searchCounter] != ',' && text[searchCounter] != '}')){
			while(text[searchCounter] != '}' && text[searchCounter] != ','){
				if(searchCounter == textLen){
					ret = Search_Result_too_long;
					break;
				}
				if(text[searchCounter] != ' ' && text[searchCounter] != '\r' && text[searchCounter] != '\n' && text[searchCounter] != '\t'){
					ret = Search_Syntax_incorrect;
					break;
				}
				searchCounter ++;
			}
		}
	}

	if(ret != Search_OK){// reset array and print message if error is found
		findError = isError;
		for(int i = 0; i< dataSize; i++){
			data[i] = 0;
		}
		if(printText == 1) {
			printf("\r\n\t%s:   \tERROR: ", tag);

			if (ret == Search_Tag_not_found){
				printf("Tag missing or incorrect syntax");
			}else if(ret == Search_Result_not_found){
				printf("Section text missing");
			}else if(ret == Search_Result_too_long){
				printf("Array is too long");
			}else if (ret == Search_Non_integers){
				printf("Unexpected Non-Integers Found");
			}else if (ret == Search_Syntax_incorrect){
				printf("Search Syntax Incorrect");
			}
		}
	}

	return ret;
}

//findJSONArrayStr not needed for current SPaRAS function
/**
* @brief  find and return an array of strings within a JSON file
* @param  const char *text text to be searched through
* @param  const char *tag tag denoting relevant string
* @param  char *data[][12] pointer of an array of strings (no longer than 12 characters) to be updated with desired variables
* @param  const int dataSize length of the array to be found
* @retval SearchErrorFlag_t ret flag to ensure there have been no errors within search
*/
/*static SearchErrorFlag_t findJSONArrayStr(const char* text, const char* tag, char data[][12], const int dataSize){
	SearchErrorFlag_t ret = Search_OK;
	int textLen = strlen(text);
	int searchCounter = 0;
	for(int i = 0; i< dataSize; i++){
		strcpy(data[i], NULL);
	}

	ret = findJSONStartPoint(text, tag, '[', &searchCounter);
	int inQuotes = 0;
	int arrayNo =0;
	while(!(text[searchCounter] ==  ']' && inQuotes == 0) && ret == Search_OK) {//loops at each element of array

		if(searchCounter == textLen){// end search after ]
			ret = Search_Result_too_long;
			break;
		}
		if(arrayNo >= dataSize){// return error if more than the expected number of array elements are found
			ret = Search_Result_too_long;
			break;
		}

		int textNo = 0;// reset position on new variable
		while(text[searchCounter] != ',' && text[searchCounter] != ']'){//loop to increase counter to start of text
			if(searchCounter == textLen){// starts counter at character after opening "
				ret = Search_Result_too_long;
				break;
			}

			if(inQuotes == 0){
				if(text[searchCounter] != ' ' && text[searchCounter] != '\t' && text[searchCounter] != '\n' && text[searchCounter] != '\r'){// return error is non character string is present
					if(!strcmp(data[arrayNo], NULL) && text[searchCounter] == '"'){
						inQuotes = 1;
					}else{
						ret = Search_Syntax_incorrect;
						break;
					}
				}
			}else{
				if(text[searchCounter] == '"' && text[searchCounter-1] != '\\'){
						inQuotes = 0;
						data[arrayNo][textNo] = '\0';
				}
				else{
					data[arrayNo][textNo] = text[searchCounter];
					textNo ++;
				}
			}
			searchCounter ++;
		}

		if(ret != Search_OK){
			break;
		}
		if(!strcmp(data[arrayNo], "\0")){// if text is empty, return error
			ret = Search_Result_not_found;
			break;
		}

		arrayNo ++;// increase non temporary variables
		if(text[searchCounter] != ']'){
			searchCounter ++;
		}
	}

	searchCounter ++;

	if(ret == Search_OK && (text[searchCounter] != ',' && text[searchCounter] != '}')){
		while(text[searchCounter] != '}' && text[searchCounter] != ','){
			if(searchCounter == textLen){
				ret = Search_Result_too_long;
				break;
			}
			if(text[searchCounter] != ' ' && text[searchCounter] != '\r' && text[searchCounter] != '\n' && text[searchCounter] != '\t'){
				ret = Search_Syntax_incorrect;
				break;
			}
			searchCounter ++;
		}
	}


	if(ret != Search_OK){// reset array and show message if error
		findError = isError;
		for(int i = 0; i< dataSize; i++){
			strcpy(data[i], "\0");
		}
		if(printText == 1) {
			printf("\r\n\t%s:   \tERROR: ", tag);

			if (ret == Search_Tag_not_found){
				printf("Tag missing or incorrect syntax");
			}else if(ret == Search_Result_not_found){
				printf("Section text missing");
			}else if(ret == Search_Result_too_long){
				printf("Array is too long");
			}else if (ret == Search_Syntax_incorrect){
				printf("Search Syntax Incorrect");
			}
		}
	}
	return ret;
}*/

//Reading and Processing Functions
/**
* @brief  find and return a sensor configuration within a JSON file
* @param  struct sensorConfigs *s the sensor configuration to be updated
* @param  char *tag The section of the JSON file to be searched through for the sensor settingstag denoting relevant string
*/
static void initialiseJSONSensorConfig(struct sensorConfigs *s, char *sensorSettings, WakeTimer_t interruptPeriod){
	char sensorSet[500];// extract the data relevant to the sensor

	if(findJSONSection(sensorSettings, s->name, sensorSet, sizeof(sensorSet)) != Search_OK){ // get section relating to desired sensor and return error if desired sensor is not included
		s->errorFlag = Sensor_Value_not_found;
	}
	else{
		if(printText == 1) printf("\r\n\t%s:", s->name);
		if(findJSONInt(sensorSet, "active", &s->active)!= Search_OK){//extract active as an integer
			s->errorFlag = Sensor_Value_not_found;
		}else if (s->active != 0 && s->active != 1) {// return error if active is out of bounds
			s->errorFlag = active_out_of_bounds;
		}
	}

	if(s->errorFlag == Sensor_OK && s->active == 1 ){

		if(!strcmp(s->name, "temperature") || !strcmp(s->name,"SHT40temperature") || !strcmp(s->name,"SHT40humidity") || !strcmp(s->name,"pressure") || !strcmp(s->name,"luminosity")){
			if(findJSONInt(sensorSet, "sampleRate", &s->sampleRate) != Search_OK){// extract sampleRate as an integer
				s->errorFlag = Sensor_Value_not_found;
			}else if(s->sampleRate <= 0) {// return error if sample rate is not a positive integer
				s->errorFlag = sampleRate_out_of_bounds;
			}else{
				s->sampleRate *= WakeTimerSecondsValue[interruptPeriod];//multiply samplerate by the current interrupt. i.e 30 minutes = 1800 seconds,
			}
		}

		if(s->errorFlag == Sensor_OK){
			if(findJSONInt(sensorSet, "recCon", &s->recordConditions) != Search_OK){//extract record conditions as an integer
				s->errorFlag = Sensor_Value_not_found;
			}else if(s->recordConditions < 1 || s->recordConditions > 4) {// return error if recordConditions are out of Bounds
				s->errorFlag = recordConditions_out_of_bounds;
			}else if(s->recordConditionsType == 1 && (s->recordConditions == 1 ||s->recordConditions == 2)) {// if record conditions type == 1 the sensor can only be declared with one threshold
				s->errorFlag = recordConditions_out_of_bounds;
			}

			if(s->errorFlag == Sensor_OK){
				if(findJSONDouble(sensorSet, "threshold1", &s->threshold1) != Search_OK){//extract threshold 1 as an integer (all settings will need threshold1)
					s->errorFlag = Sensor_Value_not_found;
				}else if(validateThresholds(s->name, s->threshold1) == isError){
					s->errorFlag = threshold1_out_of_bounds;
				}
			}
			if (s->errorFlag == Sensor_OK && s->recordConditionsType == 2 && (s->recordConditions == 1 || s->recordConditions == 2)) {// if record conditions type == 2 and if recordConditions = 3 or 4 find threshold2
				if(findJSONDouble(sensorSet, "threshold2", &s->threshold2)!= Search_OK){// extract threshold2 as an integer
					s->errorFlag = Sensor_Value_not_found;
				}else if(validateThresholds(s->name, s->threshold2) == isError){
					s->errorFlag = threshold2_out_of_bounds;
				}

				if(s->errorFlag == Sensor_OK && s->threshold1 > s->threshold2){
					int temp = s->threshold1;
					s->threshold1 = s->threshold2;
					s->threshold2 = temp;
				}
			}
		}
	}

	if(s->errorFlag != Sensor_OK){
		findError = isError;
		s->active = 0;
		s->sampleRate = 0;
		s->recordConditions = 0;
		s->threshold1 = 0;
		s->threshold2 = 0;
		if(s->errorFlag != Sensor_Value_not_found && printText == 1){
			if(s->errorFlag == active_out_of_bounds){
				printf("\r\n\tactive:   \tERROR: Active Out Of Bounds");// print the error message
			}else if(s->errorFlag == sampleRate_out_of_bounds){
				printf("\r\n\tsampleRate:   \tERROR: sampleRate Out Of Bounds");// print the error message
			}else if(s->errorFlag == recordConditions_out_of_bounds){
				printf("\r\n\trecCon:   \tERROR: recordConditions Out Of Bounds");// print the error message
			}else if(s->errorFlag == threshold1_out_of_bounds){
				printf("\r\n\tthreshold1:   \tERROR: threshold1 Out Of Bounds");// print the error message
			}else if(s->errorFlag == threshold2_out_of_bounds){
				printf("\r\n\tthreshold2:   \tERROR: threshold2 Out Of Bounds");// print the error message
			}
		}
	}
}

/**
* @brief  strip the relevant variables from the JSON Input file and flag any errors within the file's structure
* @param  char *buffText  JSON File
* @retval CodeStatus_t findError flag to ensure there have been no errors while processing the JSON file
*/
CodeStatus_t processNFCInput(char* buffText){
	// global variables to report if there is an error with the JSON file's structure
	findError = isWorking;
	int newSensorNum = 0;

	// initialise system settings
	RTC_DateTypeDef newCurrDate;
	RTC_TimeTypeDef newCurrTime;
	char newEncryptCert[35];

	//stationary medicine information
	struct staticInfo newMedInfo;
	RTC_DateTypeDef expiryDate;
	char interruptBuf[7];
	WakeTimer_t newInterrupt;
	char boardMemBuf[6];
	memStore_t newBoardMem;
	int newPrintText;
	int oldPrintText =0;

	struct sensorConfigs newAllSensorConfigs[8] = {
	{ .name = "temperature", .recordConditionsType = 2, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK},
	{ .name = "SHT40temperature", .recordConditionsType = 2, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK},
	{ .name = "SHT40humidity", .recordConditionsType = 2, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK},
	{ .name = "pressure", .recordConditionsType = 2, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK},
	{ .name = "luminosity", .recordConditionsType = 1, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK},
	{ .name = "imuacc", .recordConditionsType = 1, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK},
	{ .name = "orientation", .recordConditionsType = 2, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK},
	{ .name = "tiltrec", .recordConditionsType = 2, .active = 0, .sampleRate = 0, .recordConditions = 0, .threshold1 = 0, .threshold2 = 0, .errorFlag = Sensor_OK}
	};

	char systemSettings[400];// extract system settings data
	if(findJSONSection(buffText, "systemSettings", systemSettings, sizeof(systemSettings)) == Search_OK){// find system settings section if system settings found, extract relevant variables
		if(findJSONInt(systemSettings, "printText", &newPrintText) == Search_OK){
			oldPrintText = printText;
			printText = newPrintText;
			printf("\r\n\tprintText\t%d", printText);
		}

		if(printText == 1) printf("\r\n\n\n\n\n\n\n\n\t\tSystem Settings:\r\n");//print the Stationary Medicine Information if there are no errors, otherwise state that they haven't been found

		// interrupt period
		if (findJSONStr(systemSettings, "interruptPeriod", interruptBuf, sizeof(interruptBuf)) == Search_OK){// find interrupt
			if(!strcmp(interruptBuf,"second")){// assign interrupt definition as WakeTimer_t based on the string found in the JSON file
				newInterrupt = Second;
				if(printText == 1)printf("\r\n\tinterruptPeriod:\t%s", interruptBuf);
			}else if(!strcmp(interruptBuf,"minute")){
				newInterrupt = Minute;
				if(printText == 1)printf("\r\n\tinterruptPeriod:\t%s", interruptBuf);
			}else if(!strcmp(interruptBuf,"hour")){
				newInterrupt = Hour;
				if(printText == 1)printf("\r\n\tinterruptPeriod:\t%s", interruptBuf);
			}else if(!strcmp(interruptBuf,"day")){
				newInterrupt = Day;
				if(printText == 1)printf("\r\n\tinterruptPeriod:\t%s", interruptBuf);
			} else{
				if(printText == 1) printf("\r\n\tERROR: InterruptPeriod is Invalid");
				findError = isError;
			}
		}


		if (findJSONStr(systemSettings, "boardMem", boardMemBuf, sizeof(boardMemBuf)) == Search_OK){
			if(!strcmp(boardMemBuf,"NFC")){// assign interrupt definition as WakeTimer_t based on the string found in the JSON file
				newBoardMem = NFC;
				if(printText == 1)printf("\r\n\tboardMem:\t%s", boardMemBuf);
			}else if(!strcmp(boardMemBuf,"Flash")){
				newBoardMem = Flash;
				if(printText == 1)printf("\r\n\tboardMem:\t%s", boardMemBuf);
			}
		}

		//encryption certificate
		if (findJSONStr(systemSettings, "encryptionCert", newEncryptCert, sizeof(newEncryptCert)) == Search_OK && printText == 1) printf("\r\n\tEncryptionCert:\t%s", newEncryptCert);

		//StartDate/Time
		int tempCurrDate[3];
		if (findJSONArrayInt(systemSettings, "currDate", tempCurrDate,(sizeof(tempCurrDate)/sizeof(tempCurrDate[0]))) == Search_OK){
			if(printText == 1) printf("\r\n\tCurrDate:");
			if(evalDate(tempCurrDate, &newCurrDate) == Search_OK && printText == 1) printf("\t%d/%d/%d", newCurrDate.Date, newCurrDate.Month, newCurrDate.Year);
		}
		int tempCurrTime[3];
		if (findJSONArrayInt(systemSettings, "currTime", tempCurrTime, (sizeof(tempCurrTime)/sizeof(tempCurrTime[0]))) == Search_OK){
			if(printText == 1) printf("\r\n\tCurrTime:");
			if(evalTime(tempCurrTime, &newCurrTime) == Search_OK && printText == 1) printf("\t%d:%d:%d", newCurrTime.Hours, newCurrTime.Minutes, newCurrTime.Seconds);
		}
	}

	char medInfoBuf[300];//extract medicine info section
	if(findJSONSection(buffText, "medinfo", medInfoBuf, sizeof(medInfoBuf)) == Search_OK){//if medicine information found, extract relevant variables
		if(printText == 1) printf("\r\n\n\n\t\tMedicine Information:\r\n");//print the Stationary Medicine Information if there are no errors, otherwise state that they haven't been found

		//Medicine Information
		if (findJSONInt(medInfoBuf, "dosage", &newMedInfo.dosage) == Search_OK && printText == 1) printf("\r\n\tDosage:\t%d", newMedInfo.dosage);
		if (findJSONInt(medInfoBuf, "quantity", &newMedInfo.quantity) == Search_OK && printText == 1) printf("\r\n\tQuantity:\t%d", newMedInfo.quantity);
		int tempMedExpiry[3];
		if (findJSONArrayInt(medInfoBuf, "expiry", tempMedExpiry,(sizeof(tempMedExpiry)/sizeof(tempMedExpiry[0]))) == Search_OK){
			if(printText == 1) printf("\r\n\tExpiry:");
			if(evalDate(tempMedExpiry, &expiryDate) == Search_OK && printText == 1) printf("\t%d/%d/%d", expiryDate.Date, expiryDate.Month, expiryDate.Year);
		}
	}

	char sensorSettings[2000];// extract sensor settings
	if(findJSONSection(buffText, "sensorSettings",sensorSettings, sizeof(sensorSettings)) == Search_OK){//if sensor settings found, extract relevant variables
		if(printText == 1) printf("\r\n\n\n\t\tSensor Settings:\r\n");// use function to print the variables for each sensor's configuration

		for (int i = 0; i<numberOfSensors; i++){
			if(printText == 1) printf("\r\n");

			//read Sensor Config
			initialiseJSONSensorConfig(&newAllSensorConfigs[i], sensorSettings, newInterrupt);

			// for Tiltrec, check incompatibilities (adapted from ReadConfiguration code)
			if(i == tiltrec && newAllSensorConfigs[i].active == 1){
				if(newAllSensorConfigs[imuacc].active == 1){
					findError = isError;
					if(printText == 1)printf("\r\n\t\t ERROR: tiltrec and imuacc sensors not compatible.");
					break;
				}
				if(newAllSensorConfigs[orientation].active == 1){
					findError = isError;
					if(printText == 1)printf("\r\n\t\t ERROR: tiltrec and orientation sensors not compatible.");
					break;
				}
			}

			//print sensor config
			if(newAllSensorConfigs[i].errorFlag == Sensor_OK && printText == 1){
				if (newAllSensorConfigs[i].active == 1) {//if sensor is active without errors, print sensorConfig variables
					newSensorNum ++; // count the number of active sensors
					printf("\r\n\tactive: %d", newAllSensorConfigs[i].active);
					if(!strcmp(newAllSensorConfigs[i].name, "temperature") || !strcmp(newAllSensorConfigs[i].name,"pressure") || !strcmp(newAllSensorConfigs[i].name,"luminosity")|| !strcmp(newAllSensorConfigs[i].name,"SHT40temperature")|| !strcmp(newAllSensorConfigs[i].name,"SHT40humidity")){
						printf("\r\n\tsampleRate: %d", newAllSensorConfigs[i].sampleRate);
					}
					printf("\r\n\trecordCondition: %d", newAllSensorConfigs[i].recordConditions);
					printf("\r\n\tThreshold1: %f", newAllSensorConfigs[i].threshold1);
					if (newAllSensorConfigs[i].recordConditions == 1 || newAllSensorConfigs[i].recordConditions == 2) {// only print threshold2 if required
						printf("\r\n\tThreshold2: %f", newAllSensorConfigs[i].threshold2);
					}
				}
				else{// dont print variables if inactive
					printf("\tInactive");
				}
			}
		}
	}

	//if error, print error message, else update global variables
	if(findError == isError){
		printf("\r\n\n\n\t No Valid Configuration Found");
		printText = oldPrintText;

		LogDefinition.VirtualSensorsNum = 0;
		for(int SensorNum = 0; SensorNum < SMARTAG2_VIRTUAL_SENSORS_NUM; SensorNum++) {
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[SensorNum];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 0;
			ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = 0;
		}
	}
	else{
		printf("\r\n\n\n\t Valid Configuration Found\r\n\n");

		// from CheckIfNewConfiguration() and ReadConfiguration()
		LogDefinition.VirtualSensorsNum = (uint8_t)newSensorNum; // update virtual sensor number

		struct tm newcurTime; // update start datetime (epoch)
		newcurTime.tm_year  = newCurrDate.Year + 100;
		newcurTime.tm_mday  = newCurrDate.Date;
		newcurTime.tm_mon   = newCurrDate.Month - 1;
		/* mktime doesn't look tm_wday */
		//currTime.tm_wday = CurDate.WeekDay;
		newcurTime.tm_hour  = newCurrTime.Hours;
		newcurTime.tm_min   = newCurrTime.Minutes;
		newcurTime.tm_sec   = newCurrTime.Seconds;
		newcurTime.tm_isdst = 0;
		LogDefinition.StartDateTime = (uint32_t)mktime(&newcurTime);

	    /* Set the Date&Time */
	    if(STNFC_SetDateTime(LogDefinition.StartDateTime,&hrtc,&LogDefinition)==0) {
	      SMARTAG2_PRINTF("\r\nError: Setting RTC\r\n");
	    } else {
	       SMARTAG2_PRINTF("\r\nSet RTC Date&Time\r\n");
	    }

		struct tm expiryTime;
		expiryTime.tm_year  = expiryDate.Year + 100;
		expiryTime.tm_mday  = expiryDate.Date;
		expiryTime.tm_mon   = expiryDate.Month - 1;
		/* mktime doesn't look tm_wday */
		//currTime.tm_wday = CurDate.WeekDay;
		expiryTime.tm_hour  = 0;
		expiryTime.tm_min   = 0;
		expiryTime.tm_sec   = 0;
		expiryTime.tm_isdst = 0;
		newMedInfo.expiry = (uint32_t)mktime(&expiryTime);

		//update medInfo
		medInfo = newMedInfo;

		//update interrupt period
		interrupt = newInterrupt;
		boardMem = newBoardMem;

		//print all updated variables so far
		if(printText == 1){
			printf("\r\n\t\t testing updated configurations:");
			printf("\r\n\n\tLogDefinition.VirtualSensorsNum = %d", LogDefinition.VirtualSensorsNum);

			printf("\r\n\tLogDefinition.StartDateTime = %d", LogDefinition.StartDateTime);
			printf("\r\n\t dateTime: %d:%d:%d %d/%d/%d", newCurrTime.Hours, newCurrTime.Minutes, newCurrTime.Seconds, newCurrDate.Date, newCurrDate.Month, newCurrDate.Year);

			RTC_DateTypeDef testCurrDate;
			RTC_TimeTypeDef testCurrTime;
			if(HAL_RTC_GetTime(&hrtc, &testCurrTime, RTC_FORMAT_BIN) == HAL_OK){
				if(HAL_RTC_GetDate(&hrtc, &testCurrDate, RTC_FORMAT_BIN) == HAL_OK){
					printf("\r\n\tRTC: day/month/year = %d/%d/%d", testCurrDate.Date, testCurrDate.Month, testCurrDate.Year);
					printf("\r\n\tRTC: hours:minutes:seconds = %d:%d:%d", testCurrTime.Hours, testCurrTime.Minutes, testCurrTime.Seconds);
				}else{
					printf("\r\n\t Error Reading DateTime");
				}
			}else{
				printf("\r\n\t Error Reading DateTime");
			}


			printf("\r\n\n\t Dosage: %d", medInfo.dosage);
			printf("\r\n\t Quantity: %d", medInfo.quantity);

			struct tm testExpiryTime;
			memcpy(&testExpiryTime, localtime(&medInfo.expiry), sizeof(struct tm));
			printf("\r\n\t Expiry dateTime: %d/%d/%d", testExpiryTime.tm_mday, testExpiryTime.tm_mon+1, testExpiryTime.tm_year-100);
		}

		// Disable all the Previous Enabled Virtual Sensors
		int32_t SensorNum;
		SMARTAG2_PRINTF("\r\n\n\tRestart the Log\r\n");
		for(SensorNum = 0; SensorNum < SMARTAG2_VIRTUAL_SENSORS_NUM; SensorNum++) {
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[SensorNum];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 0;
			ConfiguratedVirtualSensorsArray[SensorNum]->SampleDeltaDateTime = 0;
		}

		// Update all configs enabled in JSON file
		SensorNum = 0;
		if(newAllSensorConfigs[temperature].active == 1){		//Temperature
			printf("\r\n\n\tTemperature Sensor Updated:");
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[STTS22H_VS_ID];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
			ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = (uint32_t)newAllSensorConfigs[temperature].sampleRate;
			ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[temperature].recordConditions-1);
			ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (uint16_t)STTS22H_SAMPLE_TO_CODED(newAllSensorConfigs[temperature].threshold1);

			SMARTAG2_PRINTF("\r\n\tSampleRate=%d", ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate);
			SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
			SMARTAG2_PRINTF("\r\n\tTh1.Ui16Value=%f", STTS22H_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value));
			if (ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType < TH_LESS) {
				ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value = (uint16_t)STTS22H_SAMPLE_TO_CODED(newAllSensorConfigs[temperature].threshold2);
				SMARTAG2_PRINTF("\r\n\tTh2.Ui16Value=%f", STTS22H_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value));
			}
			SensorNum++;
		}

		if(newAllSensorConfigs[pressure].active == 1){			//Pressure
			printf("\r\n\n\tPressure Sensor Updated:");
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[LPS22DF_VS_ID];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
			ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = (uint32_t)newAllSensorConfigs[pressure].sampleRate;
			ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[pressure].recordConditions-1);
			ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (uint16_t)LPS22DF_SAMPLE_TO_CODED(newAllSensorConfigs[pressure].threshold1);

			SMARTAG2_PRINTF("\r\n\tSampleRate=%d", ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate);
			SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
			SMARTAG2_PRINTF("\r\n\tTh1.Ui16Value=%f", LPS22DF_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value));
			if (ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType < TH_LESS) {
				ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value = (uint16_t)LPS22DF_SAMPLE_TO_CODED(newAllSensorConfigs[pressure].threshold2);
				SMARTAG2_PRINTF("\r\n\tTh2.Ui16Value=%f", LPS22DF_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value));
			}
			SensorNum++;
		}

		if(newAllSensorConfigs[luminosity].active == 1){		//Luminosity
			printf("\r\n\n\tLuminosity Sensor Updated:");
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[VD6283_LUX_VS_ID];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
			ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = (uint32_t)newAllSensorConfigs[luminosity].sampleRate;
			ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[luminosity].recordConditions-1);
			ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui32Value = (uint32_t)VD6283_LUX_SAMPLE_TO_CODED(newAllSensorConfigs[luminosity].threshold1);

			SMARTAG2_PRINTF("\r\n\tSampleRate=%d", ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate);
			SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
			SMARTAG2_PRINTF("\r\n\tTh1.Ui32Value=%f", VD6283_LUX_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui32Value));
			SensorNum++;
		}

		if(newAllSensorConfigs[imuacc].active == 1){			//Accelerometer
			printf("\r\n\n\tImuAcc Sensor Updated:");
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[LSM6DSOX32_VS_ID];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
			ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[imuacc].recordConditions-1);
			ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (uint16_t)newAllSensorConfigs[imuacc].threshold1;

			SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
			SMARTAG2_PRINTF("\r\n\tTh1.Ui16Value=%d", ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value);
			SensorNum++;
		}

		if(newAllSensorConfigs[orientation].active == 1){		//Orientation
			printf("\r\n\n\tOrientation Sensor Updated:");
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[LSM6DSOX32_6D_VS_ID];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
			ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[orientation].recordConditions-1);
			ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value = (uint8_t)newAllSensorConfigs[orientation].threshold1;

			SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
			SMARTAG2_PRINTF("\r\n\tTh1.Ui8Value=%d", ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value);
			if (ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType < TH_LESS) {
				ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value = (uint8_t)newAllSensorConfigs[orientation].threshold2;
				SMARTAG2_PRINTF("\r\n\tTh2.Ui8Value=%d", ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value);
			}
			SensorNum++;
		}

		if(newAllSensorConfigs[tiltrec].active == 1){			//Tiltrec
			printf("\r\n\n\tTiltRec Sensor Updated:");
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[LSM6DSOX32_MLC_VS_ID];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
			ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[tiltrec].recordConditions-1);
			ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value = (uint8_t)newAllSensorConfigs[tiltrec].threshold1;

			SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
			SMARTAG2_PRINTF("\r\n\tTh1.Ui8Value=%d", ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value);
			if (ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType < TH_LESS) {
				ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value = (uint8_t)newAllSensorConfigs[tiltrec].threshold2;
				SMARTAG2_PRINTF("\r\n\tTh2.Ui8Value=%d", ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value);
			}
			SensorNum++;
		}
		if(newAllSensorConfigs[SHT40temperature].active == 1){	//SHT40Temperature
			printf("\r\n\n\tSHT40 Temperature Sensor Updated:");
			ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[SHT40_TEMP_VS_ID];
			ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
			ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = (uint32_t)newAllSensorConfigs[SHT40temperature].sampleRate;
			ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[SHT40temperature].recordConditions-1);
			ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value = (uint16_t)SHT40_TEMP_SAMPLE_TO_CODED(newAllSensorConfigs[SHT40temperature].threshold1);

			SMARTAG2_PRINTF("\r\n\tSampleRate=%d", ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate);
			SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
			SMARTAG2_PRINTF("\r\n\tTh1.Ui16Value=%f", SHT40_TEMP_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui16Value));
			if (ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType < TH_LESS) {
				ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value = (uint16_t)SHT40_TEMP_SAMPLE_TO_CODED(newAllSensorConfigs[SHT40temperature].threshold2);
				SMARTAG2_PRINTF("\r\n\tTh2.Ui16Value=%f", SHT40_TEMP_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui16Value));
			}
			SensorNum++;
		}

		if(newAllSensorConfigs[SHT40humidity].active == 1){		//SHT40Humidity
				printf("\r\n\n\tSHT40 Humidity Sensor Updated:");
				ConfiguratedVirtualSensorsArray[SensorNum] = &AllVirtualSensorsArray[SHT40_HUM_VS_ID];
				ConfiguratedVirtualSensorsArray[SensorNum]->Enable = 1;
				ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate = (uint32_t)newAllSensorConfigs[SHT40humidity].sampleRate;
				ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType = (newAllSensorConfigs[SHT40humidity].recordConditions-1);
				ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value = (uint8_t)SHT40_HUM_SAMPLE_TO_CODED(newAllSensorConfigs[SHT40humidity].threshold1);

				SMARTAG2_PRINTF("\r\n\tSampleRate=%d", ConfiguratedVirtualSensorsArray[SensorNum]->SampleRate);
				SMARTAG2_PRINTF("\r\n\tThsUsageType=%s", ThresholdsUsageName[ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType]);
				SMARTAG2_PRINTF("\r\n\tTh1.Ui8Value=%f", SHT40_HUM_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th1.Ui8Value));
				if (ConfiguratedVirtualSensorsArray[SensorNum]->ThsUsageType < TH_LESS) {
					ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value = (uint8_t)SHT40_HUM_SAMPLE_TO_CODED(newAllSensorConfigs[SHT40humidity].threshold2);
					SMARTAG2_PRINTF("\r\n\tTh2.Ui8Value=%f", SHT40_HUM_CODED_TO_SAMPLE(ConfiguratedVirtualSensorsArray[SensorNum]->Th2.Ui8Value));
				}
				SensorNum++;
			}

		printf("\r\n\n\n");
	}
	return findError;
}

/**
* @brief strip the NFC Input from the NFC memory into a string, validate the string is the desired filetype and all characters within are valid
* @param  char *buffText  JSON File
* @param  char *newConfigID  ID string assigned to sensor profile
* @retval NFCDataType_t NFCDataType shows if the structure of the NFC Input is valid
*/
NFCDataType_t ReadNFC(char *buffText, char *newConfigID){
	uint8_t DataBuf8;
	NFCDataType_t NFCDataType = NFCData_Untested;// data begins untested for JSON file
	int JSONInputPrologLen = strlen(JSONInputProlog)+5;
	char bytes[2000];// in function buffer
	char buffConfigID[16];
	int bytesPosition = 0;
	int endLoop = 0;
	int inQuotes = 0;
	int bracketLayers = 0;
	int sizeOfBytes = sizeof(bytes);


	if(printText == 1) printf("\r\n\n\t Reading NFC Memory: \r\n\n\t\t");
	while(bytesPosition < sizeOfBytes) {
		if(BSP_NFCTAG_ReadData(BSP_NFCTAG_INSTANCE, (uint8_t *)&DataBuf8, (bytesPosition+SMARTAG2_START_ADDR_OFFSET), 1 )!=NFCTAG_OK){// flag error and break loop if error in reading nfc
			if(printText == 1) printf("\r\n\n\tNFC Tag ReadError\r");
			NFCDataType = NFCData_ReadError;
			break;
		}

		if(!(DataBuf8 >0) && (DataBuf8 <128)){// if byte isn't an ASCII value break
			NFCDataType = NFCData_OutofBoundsChars;
			break;
		}

		bytes[bytesPosition] = DataBuf8;// add byte/char to buffer
		if(NFCDataType == NFCData_Untested && bytesPosition > JSONInputPrologLen+1){// when the string is longer than both prologs, search for presence of datatype identifiers.
				bytes[bytesPosition +1] = '\0';

				if(strstr(bytes, JSONInputProlog)){
					NFCDataType = NFCData_JSON;
					if(printText == 1) printf("\r\t DataType: JSON\r\n");

					for(int i = 0; i < bytesPosition; i++){// if the file is .json, find the current layer that the loop is in to keep track of the file ending
						if(inQuotes == 0){// brackets only count when text is not in quotes
							if (bytes[i] == '{'){
								bracketLayers ++;
							}else if (bytes[i] == '}'){
								bracketLayers -=1;

							}else if(bytes[i] == '"'){// speech marks flip quotes
								inQuotes = 1;
							}

						}else{
							if(bytes[i] == '"' && bytes[i-1] != '\\'){// speech marks to exit quotes only count without escape character '/'
								inQuotes = 0;
							}
						}
					}

				}else{
					NFCDataType = NFCData_TypeNotFound;// if data isn't JSON, flag error and break loop
					if(printText == 1) printf("\r\t DataType: NotFound");
					break;
				}

		}else if (NFCDataType == NFCData_JSON){// keeping track of the bracket layers. the file will end when all layers have been exited.
			if(inQuotes == 0){// brackets only count when text is not in quotes
				if (bytes[bytesPosition] == '{'){
					bracketLayers ++;
				}else if (bytes[bytesPosition] == '}'){
					bracketLayers -=1;
				}else if(bytes[bytesPosition] == '"'){// speech marks flip quotes
					inQuotes = 1;
				}

			}else{
				if(bytes[bytesPosition] == '"' && bytes[bytesPosition-1] != '\\'){// speech marks to exit quotes only count without escape character '/'
					inQuotes = 0;
				}
			}
			if(bracketLayers == 0){
				endLoop = 1;
			}

		}

		if(endLoop ==1){
			break;
		}

		bytesPosition ++;
	}


	bytes[bytesPosition+1] = '\0';

	switch (NFCDataType)//reporting to console the outcome of attempting to read the file
	{
	case NFCData_JSON:
		if(printText == 1) printf("\r\n\n\t JSON File Received: \r\n\n %s\r\n\n", bytes);
		strcpy(buffText, bytes);
		if (findJSONStr(buffText, "ConfigID", buffConfigID, sizeof(buffConfigID)) == Search_OK){
			strcpy(newConfigID, buffConfigID);
			printf("\r\n\n\n\n\n\n\n\n\tConfigID: \t%s",newConfigID);
		}else{
			printf("\r\n\n\n\n\n\n\n\n\tERROR: \tNo ConfigID Found", newConfigID);
			NFCDataType = NFCData_TypeNotFound;
		}

		break;
	case NFCData_Untested:
		printf("\r\n\n\t ERROR: Written Memory too short for acceptable FileType\r\n");
		break;
	case NFCData_TypeNotFound:
		printf("\r\n\n\t ERROR: NFC Memory not in an acceptable Input Format\r\n\n");
		break;
	case NFCData_ReadError:
		printf("\r\n\n\t ERROR: Error Reading NFC Memory\r\n\n");
		break;
	case NFCData_OutofBoundsChars:
		printf("\r\n\t MemoryType Not JSON \r\n");
		break;
	}

	if(NFCDataType != NFCData_JSON){// if an error has occurred reading the data, reset the NFC Memory
		uint8_t DataBuf32 = 0;
        for (int i = 0; i < bytesPosition; i += 4) {
        	BSP_NFCTAG_WriteData(BSP_NFCTAG_INSTANCE, (uint8_t*)&DataBuf32, i, 4);
        }
	}
	return NFCDataType;
}

